/***************************************************************************
 *
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "ButtonSubdivision.h"
#include "Log.h"
#include "Subdivision.h"
#include "WindowSystems/ButtonInputEventDispatcher.h"
#include "WindowSystems/WaylandEvdevInputEvent.h"

using namespace InputEventProcessing;
using namespace LayerManagerCalibration;

ButtonInputEventDispatcher::ButtonInputEventDispatcher()
    : InputEventDispatcher()
{
}

ButtonInputEventDispatcher::~ButtonInputEventDispatcher()
{
}

void ButtonInputEventDispatcher::processTouchEvent(Subdivision* subdivision,
                                                   struct evdev_input_device* device,
                                                   coordinate& calibratedCoordinate,
                                                   struct TouchEvent& touchEvent)
{
    LOG_DEBUG("ButtonInputEventDispatcher",
              "Process touch event for subdivision, "
              "name=" << subdivision->getName() << ", "
              "device name=" << device->deviceName << ", "
              "coordinate x=" << calibratedCoordinate.x << ", "
              "y=" << calibratedCoordinate.y << ", "
              "device name=" << device->deviceName);

    // obtain the screen resolution
    WaylandEvdevInputEvent *inputEvent = static_cast<WaylandEvdevInputEvent*>(device->master);
    if ((inputEvent == NULL) || (getListener() == NULL))
    {
        LOG_WARNING("ButtonInputEventDispatcher",
                    "Failed to process touch event");
    }
    else
    {
        ButtonSubdivision *button = dynamic_cast<ButtonSubdivision*>(subdivision);

        if (touchEvent.state == INPUT_STATE_PRESSED)
        {
            if (!getState().getSubdivisionPressed(device, subdivision))
            {
                uint timeOfLastActivation = 0;

                #ifdef WL_OMIT_GETTIME
                    uint currentTime = touchEvent.time;
                #else
                    uint currentTime = getCurrentTime();
                #endif

                if (getState().getTimeOfLastActivation(device,
                                                       subdivision,
                                                       timeOfLastActivation))
                {
                    if (((currentTime - timeOfLastActivation)
                          > (button->getMinimumRepetitionInterval() * 1000))
                        ||
                        (timeOfLastActivation == currentTime))
                    {
                        getState().setSubdivisionSlotPressed(device,
                                                            subdivision,
                                                            touchEvent.slot,
                                                            true);

                        getListener()->generateKeyEvent(device,
                                                        currentTime,
                                                        button->getKeyCode(),
                                                        WL_KEYBOARD_KEY_STATE_PRESSED,
                                                        false);

                        getState().setTimeOfLastActivation(device,
                                                           subdivision,
                                                           currentTime);

                        LOG_DEBUG("ButtonInputEventDispatcher",
                                  "Button pressed at time=" << currentTime << ", "
                                  "time of last activation="
                                  << timeOfLastActivation << ", "
                                  "key code=" << button->getKeyCode());

                        if ((timeOfLastActivation == currentTime)
                            || (timeOfLastActivation > currentTime))
                                LOG_DEBUG("ButtonInputEventDispatcher",
                                          "Time error, current time="
                                          << currentTime << ", "
                                          "time of last press="
                                          << timeOfLastActivation);
                    }
                    else
                    {
                        LOG_DEBUG("ButtonInputEventDispatcher",
                                  "Button Press Attempted at time="
                                  << currentTime << ", "
                                  "Last press time=" << timeOfLastActivation
                                  << ", button repetition interval (s)="
                                  << (button->getMinimumRepetitionInterval()
                                     * 1000));
                    }
                }
                else
                {
                    getState().setSubdivisionSlotPressed(device,
                                                         subdivision,
                                                         touchEvent.slot,
                                                         true);

                    getListener()->generateKeyEvent(device,
                                                    currentTime,
                                                    button->getKeyCode(),
                                                    WL_KEYBOARD_KEY_STATE_PRESSED,
                                                    false);

                    getState().setTimeOfLastActivation(device,
                                                       subdivision,
                                                       currentTime);

                    LOG_DEBUG("ButtonInputEventDispatcher",
                              "Button pressed at time=" << currentTime << ", "
                              "key code=" << button->getKeyCode());

                    if ((timeOfLastActivation == currentTime)
                        || (timeOfLastActivation > currentTime))
                            LOG_DEBUG("ButtonInputEventDispatcher",
                                      "Time error, current time="
                                      << currentTime
                                      << ", time of last press="
                                      << timeOfLastActivation);
                }
            }
        }

        if (touchEvent.state == INPUT_STATE_RELEASED)
        {
            if (getState().getSubdivisionSlotPressed(device,
                                                     subdivision,
                                                     touchEvent.slot))
            {
                getState().setSubdivisionSlotPressed(device,
                                                     subdivision,
                                                     touchEvent.slot,
                                                     false);

                getListener()->generateKeyEvent(device,
                                                getCurrentTime(),
                                                button->getKeyCode(),
                                                WL_KEYBOARD_KEY_STATE_RELEASED,
                                                false);

                LOG_DEBUG("ButtonInputEventDispatcher",
                          "Button released, "
                          "key code=" << button->getKeyCode() << ", "
                          "device name=" << device->deviceName);
            }
        }
    }
}

void ButtonInputEventDispatcher::processEnter(LayerManagerCalibration::Subdivision* subdivision,
                                              struct evdev_input_device* device,
                                              LayerManagerCalibration::coordinate& calibratedCoordinate,
                                              struct TouchEvent& touchEvent)
{
    (void) calibratedCoordinate;
    (void) touchEvent;
    LOG_DEBUG("ButtonInputEventDispatcher",
              "Subdivision name=" << subdivision->getName() << ", "
              "device name=" << device->deviceName);
}

void ButtonInputEventDispatcher::processLeave(LayerManagerCalibration::Subdivision* subdivision,
                                              struct evdev_input_device* device,
                                              LayerManagerCalibration::coordinate& calibratedCoordinate,
                                              struct TouchEvent& touchEvent)
{
    (void) calibratedCoordinate;
    LOG_DEBUG("ButtonInputEventDispatcher",
              "Subdivision name=" << subdivision->getName() << ", "
              "device name=" << device->deviceName);

    if (getState().getSubdivisionPressed(device, subdivision))
    {

        getState().setSubdivisionSlotPressed(device,
                                             subdivision,
                                             touchEvent.slot,
                                             false);

        // Send another button press, as this is the only way we can indicate
        // that we have left the button without releasing the touch
        if (getListener() != NULL)
        {
            ButtonSubdivision *button = dynamic_cast<ButtonSubdivision*>(subdivision);

            LOG_DEBUG("ButtonInputEventDispatcher",
                      "Left subdivision, name=" << subdivision->getName()
                      << " without releasing touch, sending key press, code="
                      << button->getKeyCode());

            getListener()->generateKeyEvent(device,
                                            getCurrentTime(),
                                            button->getKeyCode(),
                                            WL_KEYBOARD_KEY_STATE_PRESSED,
                                            true);
        }
    }
}
